// Get references to DOM elements
const arena = document.getElementById('arena');
const music = document.getElementById('music');
const stopSfx = document.getElementById('stopSfx');
const burstSfx = document.getElementById('burstSfx');
const confettiSfx = document.getElementById('confettiSfx');
const status = document.getElementById('status');
const timer = document.getElementById('timer');
const tapPrompt = document.getElementById('tapPrompt');

// Game state variables
let timeLeft = 180; // Initial time left in seconds
let gameOver = false; // Flag to indicate if the game is over
let angleOffset = 0; // Used for player rotation animation
let players = []; // Array to store player DOM elements
let chairs = []; // Array to store chair DOM elements
let names = []; // Array to store player names

/**
 * Creates player and chair elements and adds them to the arena.
 * Resets the arena content before creating new elements.
 */
function createPlayersAndChairs() {
  players = []; // Clear existing players
  chairs = []; // Clear existing chairs
  // Reset arena to its initial state (dashed circle)
  arena.innerHTML = '<div class="absolute top-1/2 left-1/2 w-[80vmin] h-[80vmin] border-2 border-dashed border-gray-600 rounded-full transform -translate-x-1/2 -translate-y-1/2"></div>';

  // Create player elements based on the 'names' array
  names.forEach((name, i) => {
    const p = document.createElement('div');
    p.className = `player absolute border-2 border-white rounded-full w-[18vmin] h-[18vmin] flex items-center justify-center font-bold text-black text-lg transition-all duration-200 overflow-hidden`;
    p.id = `player${i}`; // Unique ID for each player
    p.dataset.name = name; // Store player name in a data attribute

    // Create an image element for the player's avatar
    const img = document.createElement('img');
    img.src = `https://eservices.sskassociate.in/img/${i + 1}.png`; // Placeholder image URL
    img.className = 'w-full h-full object-cover rounded-full';
    p.appendChild(img); // Add image to player div

    arena.appendChild(p); // Add player to the arena
    players.push(p); // Add player to the players array
  });

  // Create chair elements (one less than the number of players)
  for (let i = 0; i < names.length - 1; i++) {
    const c = document.createElement('div');
    c.className = `chair absolute bg-yellow-400 border-2 border-white rounded-full w-[12vmin] h-[12vmin] flex items-center justify-center text-xl transition-all duration-200`;
    c.id = `chair${i}`; // Unique ID for each chair
    c.textContent = '🪑'; // Chair emoji
    arena.appendChild(c); // Add chair to the arena
    chairs.push(c); // Add chair to the chairs array
  }
}

/**
 * Moves active players in a circular motion around the arena.
 */
function movePlayers() {
  const R = 40; // Radius of the circular path in percentage of arena size
  angleOffset += 1; // Increment angle for rotation
  const active = players.filter(p => !p.classList.contains('hidden')); // Get only visible players

  active.forEach((p, i) => {
    // Calculate angle for each player to distribute them evenly
    const angle = 2 * Math.PI * i / active.length + angleOffset * Math.PI / 180;
    // Calculate X and Y coordinates based on angle and radius
    const x = 50 + R * Math.cos(angle);
    const y = 50 + R * Math.sin(angle);
    // Set position using CSS properties
    p.style.left = `${x}%`;
    p.style.top = `${y}%`;
    p.style.transform = 'translate(-50%, -50%)'; // Center the player div
  });
}

/**
 * Places visible chairs in a circular arrangement in the center of the arena.
 */
function placeChairs() {
  const visible = chairs.filter(c => !c.classList.contains('hidden')); // Get only visible chairs
  // Adjust radius based on the number of visible chairs (0 if only one chair left)
  const R = visible.length === 1 ? 0 : 15;

  visible.forEach((c, i) => {
    // Calculate angle for each chair to distribute them evenly
    const angle = 2 * Math.PI * i / visible.length;
    // Calculate X and Y coordinates based on angle and radius
    const x = 50 + R * Math.cos(angle);
    const y = 50 + R * Math.sin(angle);
    // Set position using CSS properties
    c.style.left = `${x}%`;
    c.style.top = `${y}%`;
    c.style.transform = 'translate(-50%, -50%)'; // Center the chair div
  });
}

/**
 * Starts the game timer and updates the display every second.
 * Ends the game if time runs out or game is already over.
 */
function startTimer() {
  const interval = setInterval(() => {
    if (--timeLeft <= 0 || gameOver) {
      clearInterval(interval); // Stop the timer
      endGame(); // End the game
    }
    // Format time for display (minutes:seconds)
    const m = Math.floor(timeLeft / 60);
    const s = String(timeLeft % 60).padStart(2, '0');
    timer.textContent = `🕒 Time Left: ${m}:${s}`;
  }, 1000);
}

/**
 * Main game loop. Plays music, moves players, stops music, eliminates a player,
 * and then continues the loop if there are still multiple players.
 */
function loopGame() {
  if (gameOver) return; // Stop if game is already over

  music.play(); // Start playing music
  status.textContent = '🎶 Music Playing...'; // Update status message

  const rotator = setInterval(movePlayers, 30); // Start player movement animation

  // Set a timeout for when the music stops and elimination occurs
  setTimeout(() => {
    clearInterval(rotator); // Stop player movement
    music.pause(); // Pause the music
    stopSfx.play(); // Play stop sound effect
    status.textContent = '⛔ Music Stopped! Eliminating...'; // Update status message

    eliminateOne(); // Eliminate one player
    placeChairs(); // Re-arrange chairs

    const alive = players.filter(p => !p.classList.contains('hidden')); // Get remaining players

    if (alive.length <= 1) {
      gameOver = true; // Game over if 1 or fewer players remain
      endGame(); // End the game
    } else {
      setTimeout(loopGame, 1000); // Continue to the next round after a short delay
    }
  }, 10000); // Music plays for 10 seconds
}

/**
 * Randomly eliminates one player from the active players and hides one chair.
 */
function eliminateOne() {
  const alive = players.filter(p => !p.classList.contains('hidden')); // Get currently active players
  const unlucky = alive[Math.floor(Math.random() * alive.length)]; // Select a random player to eliminate

  if (unlucky) {
    unlucky.classList.add('hidden'); // Hide the eliminated player
    burstSfx.play(); // Play a sound effect for elimination
  }

  const visibleChairs = chairs.filter(c => !c.classList.contains('hidden')); // Get currently visible chairs
  if (visibleChairs.length > 0) {
    // Hide the last visible chair (simulating removal)
    visibleChairs[visibleChairs.length - 1].classList.add('hidden');
  }
}

/**
 * Launches a confetti animation across the screen.
 */
function launchConfetti() {
  confettiSfx.play(); // Play confetti sound effect
  for (let i = 0; i < 50; i++) { // Create 50 confetti pieces
    const c = document.createElement('div');
    c.className = 'confetti';
    c.style.left = Math.random() * 100 + 'vw'; // Random horizontal position
    c.style.backgroundColor = `hsl(${Math.random()*360}, 100%, 50%)`; // Random color
    document.body.appendChild(c); // Add confetti to the body
    setTimeout(() => c.remove(), 3000); // Remove confetti after 3 seconds
  }
}

/**
 * Ends the game, declares the winner, and shows post-game options.
 */
function endGame() {
  const winner = players.find(p => !p.classList.contains('hidden')); // Find the last remaining player

  if (winner) {
    // Animate winner to the center and make them stand out
    winner.style.transition = 'all 1s ease-in-out';
    winner.style.left = '50%';
    winner.style.top = '50%';
    winner.style.transform = 'translate(-50%, -50%) scale(1.8)';
    winner.style.zIndex = '10';
    winner.style.boxShadow = '0 0 20px gold';
    winner.style.border = '3px solid gold';

    // Add a trophy icon
    const trophy = document.createElement('div');
    trophy.className = 'trophy';
    trophy.innerHTML = '🏆';
    arena.appendChild(trophy);
  }

  const name = winner ? winner.dataset.name : 'No one'; // Get winner's name or 'No one'
  status.innerHTML = `🏁 <span class="text-3xl text-yellow-400 font-bold">Winner: ${name}</span>`; // Display winner
  timer.textContent = '🛑 Game Ended'; // Update timer status
  launchConfetti(); // Launch confetti
  document.getElementById('afterGame').classList.remove('hidden'); // Show post-game buttons
}

/**
 * Resets the game to its initial state and starts a new round.
 */
function resetGame() {
  // Clear arena content except for the dashed circle
  arena.innerHTML = `<div class="absolute top-1/2 left-1/2 w-[80vmin] h-[80vmin] border-2 border-dashed border-gray-600 rounded-full transform -translate-x-1/2 -translate-y-1/2"></div>`;
  createPlayersAndChairs(); // Recreate players and chairs
  timeLeft = 180; // Reset timer
  gameOver = false; // Reset game over flag
  angleOffset = 0; // Reset rotation offset
  status.textContent = '⏳ Restarting...'; // Update status
  timer.textContent = '🕒 Time Left: 3:00'; // Reset timer display
  document.getElementById('afterGame').classList.add('hidden'); // Hide post-game buttons

  // Remove any existing trophy elements
  const existingTrophy = arena.querySelector('.trophy');
  if (existingTrophy) {
    existingTrophy.remove();
  }

  // Start the game loop after a brief delay
  setTimeout(() => {
    placeChairs(); // Place chairs for the new round
    loopGame(); // Start the game loop
    startTimer(); // Start the timer
  }, 1000);
}

/**
 * Captures a screenshot of the entire body and allows the user to download it.
 */
function captureScreenshot() {
  html2canvas(document.body).then(canvas => {
    const link = document.createElement('a');
    link.download = 'musical-chairs-winner.png'; // Suggested filename
    link.href = canvas.toDataURL(); // Data URL of the captured image
    link.click(); // Programmatically click the link to trigger download
  });
}

// Event listener for the initial tap to start the game
document.addEventListener('click', () => {
  if (tapPrompt) tapPrompt.remove(); // Remove the "Tap to Start" prompt

  // Define initial player names
  names = ['Alice', 'Bob', 'Cathy', 'Dave', 'Ella', 'Frank'];

  let pre = 5; // Countdown for game start
  status.textContent = '⏳ Starting in 5 sec...'; // Initial countdown message

  // Countdown interval
  const countdown = setInterval(() => {
    pre--;
    if (pre <= 0) {
      clearInterval(countdown); // Stop countdown
      status.textContent = '🎮 Game Started!'; // Update status
      createPlayersAndChairs(); // Create players and chairs
      placeChairs(); // Place chairs
      startTimer(); // Start the game timer
      loopGame(); // Start the main game loop
    } else {
      status.textContent = `⏳ Starting in ${pre} sec...`; // Update countdown
    }
  }, 1000); // Update every second
}, { once: true }); // Ensure this listener only fires once
