<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Musical Chairs Game</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <script src="https://cdn.tailwindcss.com"></script>
  <script src="https://html2canvas.hertzen.com/dist/html2canvas.min.js"></script>
  <style>
    @keyframes blink { 0%, 100% {opacity: 1;} 50% {opacity: 0;} }
    @keyframes fadeOut { 0% { opacity: 1; transform: scale(1); } 100% { opacity: 0; transform: scale(0); } }
    @keyframes glow { 0%, 100% { box-shadow: 0 0 10px gold; } 50% { box-shadow: 0 0 30px #fff700; } }
    @keyframes burst { 0% { transform: scale(1); opacity: 1; } 100% { transform: translate(var(--dx), var(--dy)) scale(0.1); opacity: 0; } }
    @keyframes confettiFall { 0% { transform: translateY(-100px) rotate(0deg); } 100% { transform: translateY(100vh) rotate(720deg); } }
    .trophy {
      position: absolute;
      top: 50%;
      left: 50%;
      transform: translate(-50%, -50%) translateY(6rem);
      font-size: 3rem;
      animation: glow 1s infinite;
      z-index: 20;
    }
    .confetti {
      position: fixed;
      width: 10px;
      height: 10px;
      background: red;
      animation: confettiFall 3s linear forwards;
      opacity: 0.9;
      z-index: 9999;
    }
  </style>
</head>
<body class="bg-gradient-to-br from-gray-900 to-black text-white font-sans text-center p-4 pb-20">
  <h1 class="text-cyan-400 text-xl font-bold drop-shadow-lg">🎮 Musical Chairs</h1>
  <p id="tapPrompt" class="text-cyan-300 animate-[blink_1s_infinite]">👆 Tap to Start</p>
  <p id="timer" class="mt-2 text-sm">🕒 Time Left: 3:00</p>

  <div class="relative w-[90vmin] h-[90vmin] mx-auto mt-4" id="arena">
    <div class="absolute top-1/2 left-1/2 w-[80vmin] h-[80vmin] border-2 border-dashed border-gray-600 rounded-full transform -translate-x-1/2 -translate-y-1/2"></div>
  </div>

  <p id="status" class="mt-2 text-sm">Waiting to start...</p>
  <div id="afterGame" class="hidden mt-4 space-x-2">
    <button onclick="resetGame()" class="px-4 py-2 text-white bg-green-600 rounded-lg">🔁 Play Again</button>
    <button onclick="captureScreenshot()" class="px-4 py-2 text-white bg-blue-600 rounded-lg">📸 Save Screenshot</button>
  </div>

  <audio id="music" src="https://eservices.sskassociate.in/game-gaming-minecraft-background-music-362185.mp3" loop></audio>
  <audio id="stopSfx" src="https://cdn.pixabay.com/download/audio/2023/03/08/audio_62dfc61528.mp3?filename=notification-123739.mp3"></audio>
  <audio id="burstSfx" src="https://cdn.pixabay.com/download/audio/2022/03/31/audio_a55ff74c56.mp3" preload="auto"></audio>
  <audio id="confettiSfx" src="https://cdn.pixabay.com/download/audio/2021/09/28/audio_c55e51e96b.mp3" preload="auto"></audio>

  <script>
    const arena = document.getElementById('arena');
    const music = document.getElementById('music');
    const stopSfx = document.getElementById('stopSfx');
    const burstSfx = document.getElementById('burstSfx');
    const confettiSfx = document.getElementById('confettiSfx');
    const status = document.getElementById('status');
    const timer = document.getElementById('timer');
    const tapPrompt = document.getElementById('tapPrompt');

    let timeLeft = 180;
    let gameOver = false;
    let angleOffset = 0;
    let players = [];
    let chairs = [];
    let names = [];

    function createPlayersAndChairs() {
      players = [];
      chairs = [];
      arena.innerHTML = '<div class="absolute top-1/2 left-1/2 w-[80vmin] h-[80vmin] border-2 border-dashed border-gray-600 rounded-full transform -translate-x-1/2 -translate-y-1/2"></div>';
      names.forEach((name, i) => {
        const p = document.createElement('div');
        p.className = `player absolute border-2 border-white rounded-full w-[18vmin] h-[18vmin] flex items-center justify-center font-bold text-black text-lg transition-all duration-200 overflow-hidden`;
        p.id = `player${i}`;
        p.dataset.name = name;

        const img = document.createElement('img');
        img.src = `https://eservices.sskassociate.in/img/${i + 1}.png`;
        img.className = 'w-full h-full object-cover rounded-full';
        p.appendChild(img);

        arena.appendChild(p);
        players.push(p);
      });
      for (let i = 0; i < names.length - 1; i++) {
        const c = document.createElement('div');
        c.className = `chair absolute bg-yellow-400 border-2 border-white rounded-full w-[12vmin] h-[12vmin] flex items-center justify-center text-xl transition-all duration-200`;
        c.id = `chair${i}`;
        c.textContent = '🪑';
        arena.appendChild(c);
        chairs.push(c);
      }
    }

    function movePlayers() {
      const R = 40;
      angleOffset += 1;
      const active = players.filter(p => !p.classList.contains('hidden'));
      active.forEach((p, i) => {
        const angle = 2 * Math.PI * i / active.length + angleOffset * Math.PI / 180;
        const x = 50 + R * Math.cos(angle);
        const y = 50 + R * Math.sin(angle);
        p.style.left = `${x}%`;
        p.style.top = `${y}%`;
        p.style.transform = 'translate(-50%, -50%)';
      });
    }

    function placeChairs() {
      const visible = chairs.filter(c => !c.classList.contains('hidden'));
      const R = visible.length === 1 ? 0 : 15;
      visible.forEach((c, i) => {
        const angle = 2 * Math.PI * i / visible.length;
        const x = 50 + R * Math.cos(angle);
        const y = 50 + R * Math.sin(angle);
        c.style.left = `${x}%`;
        c.style.top = `${y}%`;
        c.style.transform = 'translate(-50%, -50%)';
      });
    }

    function startTimer() {
      const interval = setInterval(() => {
        if (--timeLeft <= 0 || gameOver) {
          clearInterval(interval);
          endGame();
        }
        const m = Math.floor(timeLeft / 60);
        const s = String(timeLeft % 60).padStart(2, '0');
        timer.textContent = `🕒 Time Left: ${m}:${s}`;
      }, 1000);
    }

    function loopGame() {
      if (gameOver) return;
      music.play();
      status.textContent = '🎶 Music Playing...';
      const rotator = setInterval(movePlayers, 30);
      setTimeout(() => {
        clearInterval(rotator);
        music.pause();
        stopSfx.play();
        status.textContent = '⛔ Music Stopped! Eliminating...';
        eliminateOne();
        placeChairs();
        const alive = players.filter(p => !p.classList.contains('hidden'));
        if (alive.length <= 1) {
          gameOver = true;
          endGame();
        } else {
          setTimeout(loopGame, 1000);
        }
      }, 10000);
    }

    function eliminateOne() {
      const alive = players.filter(p => !p.classList.contains('hidden'));
      const unlucky = alive[Math.floor(Math.random() * alive.length)];
      unlucky.classList.add('hidden');
      const visibleChairs = chairs.filter(c => !c.classList.contains('hidden'));
      if (visibleChairs.length > 0) visibleChairs[visibleChairs.length - 1].classList.add('hidden');
    }

    function launchConfetti() {
      confettiSfx.play();
      for (let i = 0; i < 50; i++) {
        const c = document.createElement('div');
        c.className = 'confetti';
        c.style.left = Math.random() * 100 + 'vw';
        c.style.backgroundColor = `hsl(${Math.random()*360}, 100%, 50%)`;
        document.body.appendChild(c);
        setTimeout(() => c.remove(), 3000);
      }
    }

    function endGame() {
      const winner = players.find(p => !p.classList.contains('hidden'));
      if (winner) {
        winner.style.transition = 'all 1s ease-in-out';
        winner.style.left = '50%';
        winner.style.top = '50%';
        winner.style.transform = 'translate(-50%, -50%) scale(1.8)';
        winner.style.zIndex = '10';
        winner.style.boxShadow = '0 0 20px gold';
        winner.style.border = '3px solid gold';

        const trophy = document.createElement('div');
        trophy.className = 'trophy';
        trophy.innerHTML = '🏆';
        arena.appendChild(trophy);
      }
      const name = winner ? winner.dataset.name : 'No one';
      status.innerHTML = `🏁 <span class="text-3xl text-yellow-400 font-bold">Winner: ${name}</span>`;
      timer.textContent = '🛑 Game Ended';
      launchConfetti();
      document.getElementById('afterGame').classList.remove('hidden');
    }

    function resetGame() {
      arena.innerHTML = `<div class="absolute top-1/2 left-1/2 w-[80vmin] h-[80vmin] border-2 border-dashed border-gray-600 rounded-full transform -translate-x-1/2 -translate-y-1/2"></div>`;
      createPlayersAndChairs();
      timeLeft = 180;
      gameOver = false;
      angleOffset = 0;
      status.textContent = '⏳ Restarting...';
      timer.textContent = '🕒 Time Left: 3:00';
      setTimeout(() => {
        placeChairs();
        loopGame();
        startTimer();
      }, 1000);
    }

    function captureScreenshot() {
      html2canvas(document.body).then(canvas => {
        const link = document.createElement('a');
        link.download = 'musical-chairs-winner.png';
        link.href = canvas.toDataURL();
        link.click();
      });
    }

    document.addEventListener('click', () => {
      if (tapPrompt) tapPrompt.remove();
      names = ['Alice', 'Bob', 'Cathy', 'Dave', 'Ella', 'Frank'];
      let pre = 5;
      status.textContent = '⏳ Starting in 5 sec...';
      const countdown = setInterval(() => {
        pre--;
        if (pre <= 0) {
          clearInterval(countdown);
          status.textContent = '🎮 Game Started!';
          createPlayersAndChairs();
          placeChairs();
          startTimer();
          loopGame();
        } else {
          status.textContent = `⏳ Starting in ${pre} sec...`;
        }
      }, 1000);
    }, { once: true });
  </script>
  <div class="bg-gray-800 text-sm text-white rounded-lg p-4 mt-4 max-w-xl mx-auto">
  <h2 class="text-cyan-400 font-bold mb-2 text-lg">📜 Game Rules</h2>
  <ul class="list-disc list-inside space-y-1 text-left">
    <li>Game starts when you tap the screen.</li>
    <li>Music plays while players move in a circle.</li>
    <li>When music stops, one player is eliminated randomly.</li>
    <li>One chair is removed in every round.</li>
    <li>The last remaining player wins the game.</li>
    <li>Winner gets a 🎉 confetti celebration and a 🏆 trophy.</li>
    <li>Use the 📸 button to save the final result.</li>
  </ul>
</div>

  <!-- 🎮 Game Footer -->
<footer class="fixed bottom-0 left-0 w-full bg-gradient-to-r from-gray-800 via-gray-900 to-gray-800 text-white shadow-inner z-50">
  <div class="flex justify-around items-center text-xs py-2 px-1">
    <div class="text-center cursor-pointer hover:text-cyan-400 transition">
      <div class="text-lg">👤</div>
      <div>Profile</div>
    </div>
    <div class="text-center cursor-pointer hover:text-cyan-400 transition">
      <div class="text-lg">💰</div>
      <div>Balance</div>
    </div>
    <div class="text-center cursor-pointer hover:text-green-400 transition">
      <div class="text-lg">➕</div>
      <div>Add</div>
    </div>
    <div class="text-center cursor-pointer hover:text-red-400 transition">
      <div class="text-lg">➖</div>
      <div>Withdraw</div>
    </div>
    <div class="text-center cursor-pointer hover:text-yellow-400 transition" onclick="resetGame()">
      <div class="text-lg">🎮</div>
      <div>Play</div>
    </div>
  </div>
</footer>


</body>
</html>
